/*
https://docs.nestjs.com/controllers#controllers
*/

import {
  Body,
  Get,
  Controller,
  Post,
  UsePipes,
  ValidationPipe,
  UseGuards,
  Put,
  Param,
} from '@nestjs/common';
import {
  ApiBadRequestResponse,
  ApiBearerAuth,
  ApiCreatedResponse,
  ApiResponse,
  ApiUnauthorizedResponse,
} from '@nestjs/swagger';
import { JwtAuthGuard } from 'src/auth/guards/jwt-auth-guard';
import { UserRegistrationRequestDto } from 'src/dto/user-registration.req.dto';
import { User } from 'src/entity/user.entity';
import { UserService } from 'src/services/user.service';
import { ApiDefaultResponses } from 'src/utils/common.utils';

@Controller('api/user')
export class UserController {
  constructor(private userDbService: UserService) {}

  @Post('/register')
  @ApiDefaultResponses('Registration completed!')
  @UsePipes(ValidationPipe)
  async doUserRegistration(
    @Body() registerUserData: UserRegistrationRequestDto,
  ): Promise<User> {
    return await this.userDbService.doUserRegistration(registerUserData);
  }

  @UseGuards(JwtAuthGuard)
  @Get('/')
  @ApiBearerAuth() //this tells swagger to use authorisation
  @ApiDefaultResponses('User fetching completed!')
  async getAllUsers(): Promise<User[]> {
    return await this.userDbService.getAllUsers();
  }

  @UseGuards(JwtAuthGuard)
  @Put('/:id')
  @UsePipes(ValidationPipe)
  @ApiBearerAuth() //this tells swagger to use authorisation
  @ApiDefaultResponses('User updation completed!')
  async updateUser(
    @Param('id') id: number,
    @Body() userData: UserRegistrationRequestDto,
  ): Promise<User> {
    return await this.userDbService.updateUser(id, userData);
  }
}
